%=====================================================================================
%
%  Standard Risk Aversion and Efficient Risk Sharing
%  By Richard M.H. Suen
%
%  The following codes are used to solve for the joint investment decisions
%  made by a CARA-utility-maximizer and a CRRA-utility-maximizer, both with 
%  and without background risk.
%
%======================================================================================

close all; 
clear all;

global lambda_1 lambda_2 sigma phi z_1

pause on

%=====================================================================================
% Agent 1 has CARA utility 
% Agent 2 has CRRA utility

phi   = 0.1;   % Parameter in CARA utility
sigma = 0.4;   % Paremeter in CRRA utility

lambda_1 = 1.5;  % Pareto weight for Agent 1
lambda_2 = 1.0;  % Pareto weight for Agent 2

% First we plot the risk tolerance and absolute prudence for the aggregate
% utility function (representative agent) over a certain range.

M = 201; 
z = linspace(0.5, 10.5, M);

T_hat = zeros(1,M);  % Risk tolerance
P_hat = zeros(1,M);  % Absolute prudence
kappa = zeros(2,M);  % Sharing rules

for i = 1:M
    
    z_1 = z(i);
    
    [sol_1, val_1, flag_1] = fsolve('SR', 0, optimoptions('fsolve','Display','off')); 
    
    %=================================================================================
    
    if ( flag_1 ~= 1 )
        
        fprintf('Error: fsolve is not converging.')
        z(i)
        pause
        
    end
    
    %=================================================================================
    
    kap_1 = sol_1;
    
    kap_2 = ((lambda_2/lambda_1)^(1/sigma))*exp(phi*kap_1/sigma);
    
    kappa(:,i) = [kap_1 kap_2];
    
    T_hat(i) = 1/phi + kap_2/sigma; 
    
    P_hat(i) = ( 1/phi + (1 + sigma)*kap_2/sigma/sigma )/T_hat(i)/T_hat(i); 
    
end

figure(1)
plot(z, T_hat);
title('Risk Tolerance');

figure(2)
plot(z, P_hat);
title('Absolute Prudence');

figure(3)
plot(z, kappa(1,:))
hold on
plot(z, kappa(2,:), 'g')
title('Sharing Rules');
hold off

%=======================================================================================

w = 4.5;           % Initial wealth
x = [-0.2 0.24];   % Payoffs of risky assets; with equal probability
y = [-2.0 0 2.6];  % Background risk; with equal probability

% We now use the bisection method to solve for the optimal portfolio choice when there is 
% no background risk.

a_max = 10;
a_min = 0;

Error = -999;
Tol   = 1e-08;

c_1 = zeros(1,2);

while ( abs(Error) > Tol )
        
    a = ( a_min + a_max )/2;
    
    for j = 1:2
        
        z_1 = w + a*x(j);
        
        [sol_2, val_2, flag_2] = fsolve('SR', 0, optimoptions('fsolve','Display','off'));
        
        %=================================================================================
        
        if ( flag_2 ~= 1 )
            
            fprintf('Error: fsolve is not converging.')
            pause
            
        end
        
        %=================================================================================
        
        c_1(j) = sol_2;
        
    end
    
    MU = exp( -phi*c_1 );
    
    Error = x(1)*MU(1)/( x(2)*MU(2) ) + 1;
    
    %===================================================================
    if ( Error > 0 )
        
        a_min = a;
        
    else
        
        a_max = a;
        
    end
    %====================================================================
    
end

Alpha_star = a; % Optimal choice without background risk

%========================================================================================
% Before we solve the optimal portfolio choice under background risk, we
% first check that the background risk will raise the representative
% agent's expected marginal utility under Alpha_star. 
%========================================================================================

d = zeros(1,6);

d(1) = w + a*x(1) + y(1);
d(2) = w + a*x(1) + y(2);
d(3) = w + a*x(1) + y(3);

d(4) = w + a*x(2) + y(1);
d(5) = w + a*x(2) + y(2);
d(6) = w + a*x(2) + y(3);

g_1 = zeros(1,6);

for n = 1:6
    
    z_1 = d(n);
    
    [sol_3, val_3, flag_3] = fsolve('SR', 0, optimoptions('fsolve','Display','off'));
    
    %===================================================================================
    
    if ( flag_3 ~= 1 )
        
        fprintf('Error: fsolve is not converging.')
        pause
        
    end
    
    %====================================================================================
    
    g_1(n) = sol_3;
    
end

MU_1 = exp( -phi*g_1 );  % Expected marginal utility under background risk
MU_2 = exp( -phi*c_1 );  % Expected marginal utility without background risk

EX01 = sum( MU_1 )/6;
EX02 = sum( MU_2 )/2;

[EX01 EX02]

%=====================================================================================
% We now use the bisection method to solve for the optimal portfolio choice
% under background risk. 
%=====================================================================================

a2_max = 10;
a2_min = 0;

Error2 = -999;
Tol    = 1e-08; 

q   = zeros(1,6);
g_2 = zeros(1,6);

while ( abs(Error2) > Tol )
    
    a2 = (a2_min + a2_max)/2; 
    
    q(1) = w + a2*x(1) + y(1);
    q(2) = w + a2*x(1) + y(2);
    q(3) = w + a2*x(1) + y(3); 
    
    q(4) = w + a2*x(2) + y(1);
    q(5) = w + a2*x(2) + y(2);
    q(6) = w + a2*x(2) + y(3); 
    
    clear i
    
    for i = 1:6
        
        z_1 = q(i);
        
        [sol_4, val_4, flag_4] = fsolve('SR', 0, optimoptions('fsolve','Display','off'));
        
        %========================================================================
        
        if ( flag_4 ~= 1 )
            
            fprintf('Error: fsolve is not converging.')
            pause
            
        end
        
        %========================================================================
        
        g_2(i) = sol_4;
        
    end
    
    MU = exp( -phi*g_2 );
    
    Error2 = x(1)*sum( MU(1:3) )/( x(2)*sum( MU(4:6) ) ) + 1;
    
    %===================================================================
    if ( Error2 > 0 )
        
        a2_min = a2;
        
    else
        
        a2_max = a2;
        
    end
    %====================================================================
    
end

Alpha_star2 = a2; % Optimal choice with background risk

Solutions = [Alpha_star Alpha_star2]
    
    
    





        
       
        
        
        
        
        


    
    
    
    

